/*********************************************************************************

Copyright (c) 2018, Vernier Software & Technology
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
	* Redistributions of source code must retain the above copyright
		notice, this list of conditions and the following disclaimer.
	* Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
	* Neither the name of Vernier Software & Technology nor the
		names of its contributors may be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL VERNIER SOFTWARE & TECHNOLOGY BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

**********************************************************************************/
#ifndef _D2PIO_LIB_INTERFACE_H_
#define _D2PIO_LIB_INTERFACE_H_

/***************************************************************************************************************************
	D2PIO_lib_interface.h 

	This file documents the 'C' interface to the D2PIO library.

	This library is implemented as D2PIO_lib.dll in Windows, libD2PIO_lib.dylib on the Mac, and libD2PIO in Linux.

	The D2PIO library provides an application program with full access to the data acquisition capabilities built
	into the Vernier Go Direct sensors.

	The D2PIO library API is fairly broad, so knowing where to start is hard. The documentation for the
	D2PIO_Device_Open() and the D2PIO_Device_SendCmdAndGetResponse() functions are a good starting place.

	The console application D2PIO_DeviceCheck gives a good overview of the D2PIO library API.

***************************************************************************************************************************/
#include "D2PIOSourceCmds.h"

#ifdef TARGET_OS_LINUX
#ifdef __cplusplus
	#define D2PIO_LIB_INTERFACE_DECL extern "C" __attribute__ ((visibility("default")))
#else
	#define D2PIO_LIB_INTERFACE_DECL __attribute__ ((visibility("default")))
#endif
#elif TARGET_OS_MAC
#ifdef __cplusplus
	#define D2PIO_LIB_INTERFACE_DECL extern "C"
#else
	#define D2PIO_LIB_INTERFACE_DECL
#endif
#else
#ifdef __cplusplus
	#define D2PIO_LIB_INTERFACE_DECL extern "C"
#else
	#define D2PIO_LIB_INTERFACE_DECL
#endif
#endif

#define D2PIO_MAX_SIZE_DEVICE_NAME 220

#ifndef GTYPE_NUMS
#define GTYPE_NUMS
#include <stdint.h>
typedef int16_t gtype_int16;
typedef uint16_t gtype_uint16;
typedef int32_t gtype_int32;
typedef uint32_t gtype_uint32;
typedef int64_t gtype_int64;
typedef uint64_t gtype_uint64;
typedef unsigned char gtype_bool;
typedef double gtype_real64;
typedef float gtype_real32;
#endif

typedef void *D2PIO_LIBRARY_HANDLE;
typedef void *D2PIO_DEVICE_HANDLE;
typedef void *D2PIO_DEVICE_LIST_HANDLE;
typedef void *D2PIO_PTR;

#define D2PIO_OPERATION_IN_PROGRESS -200
#define D2PIO_ERROR_BLE_RADIO_NOT_AVAILABLE -300

//List of communication transport types.
#define D2PIO_COMM_TRANSPORT_UNKNOWN 0
#define D2PIO_COMM_TRANSPORT_USB 1
#define D2PIO_COMM_TRANSPORT_SERIAL 2
#define D2PIO_COMM_TRANSPORT_BLUETOOTH 3
#define D2PIO_COMM_TRANSPORT_BLUETOOTH_LE 4
#define D2PIO_COMM_TRANSPORT_WIRELESS_80211_X 5
#define D2PIO_COMM_TRANSPORT_HARDWIRE_LAN 6

#define D2PIO_DEVTYPE_UNKNOWN 0
#define D2PIO_DEVTYPE_GENERIC 30

//Following devices are not officially supported:
#define D2PIO_DEVTYPE_GO_WIRELESS_TEMPERATURE 21
#define D2PIO_DEVTYPE_GO_WIRELESS_PH 22
#define D2PIO_DEVTYPE_GO_WIRELESS_EA 23
#define D2PIO_DEVTYPE_GO_WIRELESS_LINK 24
#define D2PIO_DEVTYPE_POLAR_HEART_RATE 25

#define D2PIO_TIMEOUT_MS_DEFAULT (2000*4) // TODO: (bre) GWHR demo
#define D2PIO_USB_OPEN_TIMEOUT_MS 3000
#define D2PIO_BLE_OPEN_TIMEOUT_MS 25000
#define D2PIO_INIT_CMD_RESPONSE_TIMEOUT_MS 2000
#define D2PIO_TIMEOUT_MS_READ_DDSMEMBLOCK 2000
#define D2PIO_TIMEOUT_MS_WRITE_DDSMEMBLOCK 4000

#ifdef TARGET_PLATFORM_NACL
struct tagD2PIO_NaCl_libParms
{
	char deviceServerName[D2PIO_MAX_SIZE_DEVICE_NAME];
	gtype_uint16 usbDeviceServerPort;
	gtype_uint16 bluetoothDeviceServerPort;
};
typedef struct tagD2PIO_NaCl_libParms D2PIO_NaCl_libParms;
#endif

/***************************************************************************************************************************
Function Name: D2PIO_Init()

Purpose:	Call D2PIO_Init() once before making any other D2PIO function calls.
			D2PIO_Init() and D2PIO_Uninit() should be called from the same thread.

Return:		Handle to D2PIO library if successful, else NULL.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL D2PIO_LIBRARY_HANDLE D2PIO_Init(
	gtype_bool bInitUSB,	//[in]
	gtype_bool bInitBLE,	//[in]
	D2PIO_PTR pOS_libParms, //[in] ptr to OS specific parms, usually NULL
	gtype_uint32 parmsLen);	//[in] length of OS specific parms, usually zero

/***************************************************************************************************************************
Function Name: D2PIO_Uninit()

Purpose:	Call D2PIO_Uninit() once to 'undo' D2PIO_Init().
			D2PIO_Init() and D2PIO_Uninit() should be called from the same thread.

Return:		0 iff successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Uninit(D2PIO_LIBRARY_HANDLE hLib);//[in] handle to library returned by D2PIO_Init().

/***************************************************************************************************************************
Function Name: D2PIO_Diags_SetDebugTraceThreshold()

Purpose:	D2PIO lib generates a variety of debugging messages when it runs. Each message is assigned a severity
			when it is generated. Only messages that are assigned a priority >= the debug trace threshold are actually
			sent to the debug output. Call D2PIO_Diags_SetDebugTraceThreshold(D2PIO_TRACE_SEVERITY_LOWEST) for max
			debug output.

			On windows systems, these messages are passed to the OutputDebugString() function.
			On Mac and Linux systems, these messages are sent to STDOUT and/or STDERR.

Return:		0 iff successful, else -1.

****************************************************************************************************************************/
#define D2PIO_TRACE_SEVERITY_LOWEST 1
#define D2PIO_TRACE_SEVERITY_LOW 10
#define D2PIO_TRACE_SEVERITY_MEDIUM 50
#define D2PIO_TRACE_SEVERITY_HIGH 100
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Diags_SetDebugTraceThreshold(
	D2PIO_LIBRARY_HANDLE hLib,
	gtype_int32 threshold);//[in] Only trace messages marked with a severity >= threshold are actually sent to the debug output.

D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Diags_GetDebugTraceThreshold(
	D2PIO_LIBRARY_HANDLE hLib,
	gtype_int32 *pThreshold);//[out]

D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_TimeSyncAllDevices(D2PIO_LIBRARY_HANDLE hLib, gtype_uint32 timeoutMs);

/***************************************************************************************************************************
Function Name: D2PIO_GetDLLVersion()

Purpose:	This routine returns the major and minor version numbers for the instance of the D2PIO library that is
			currently running.

			If a function is not guaranteed to be present in all supported versions of the D2PIO library, then the line
			"Added in version 'major.minor'" will appear in the function description in this file.

			It is our intention that all versions of the D2PIO library created subsequent to a given version, will be
			backwards compatible with the older version. You should be able to replace an old version of the D2PIO library
			with a newer version and everything should still work without rebuilding your application.

			Note that version major2.minor2 is later than version major1.minor1
			iff. ((major2 > major1) || ((major2 == major1) && (minor2 > minor1))).

			Backwards compatibility is definitely our intention, but we do not absolutely guarantee it. If you think
			that you have detected a backwards compatibility bug, then please report it to Vernier Software & Technology.
			Calling D2PIO_GetDLLVersion() from your application is a way to identify precisely which version of
			the D2PIO library you are actually using.

Return:		0 iff successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_GetLibVersion(
	D2PIO_LIBRARY_HANDLE hLib,	//[in] handle returned by D2PIO_Init()
	gtype_uint16 *pMajorVersion, //[out]
	gtype_uint16 *pMinorVersion); //[out]

/***************************************************************************************************************************
Function Name: D2PIO_SearchForDevices()

Purpose:	The D2PIO library maintains a separate list of available devices for each supported device type.
			D2PIO_SearchForDevices() updates the list for the specified device type.

			Currently, D2PIO_SearchForDevices() only works for D2PIO_COMM_TRANSPORT_USB.
			Use D2PIO_EnableAutoDeviceDiscoveryAcrossCommTransport() for D2PIO_COMM_TRANSPORT_BLUETOOTH_LE.

			Call D2PIO_OpenDeviceListSnapshot() to access the list of discovered devices.

Return:		0 iff successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_SearchForDevices(
	D2PIO_LIBRARY_HANDLE hLib,		//[in] handle returned by D2PIO_Init()
	gtype_uint32 deviceType,		//[in] D2PIO_DEVTYPE_...
	gtype_uint32 commTransportId,	//[in] D2PIO_COMM_TRANSPORT_...
	D2PIO_PTR pParams,				//[in] ptr to context specific search params. Not currently used.
	gtype_uint32 *pDeviceListSignature);//[out] ptr to device list signature. Signatures are not fully implemented yet.

typedef gtype_int32(*P_D2PIO_DEVICELISTNOTIFICATION_CALLBACK)
(
	D2PIO_PTR pNotification,			//Typically ptr to D2PIODevicePlugNotification.
	gtype_uint32 numBytes,				//Typically sizeof(D2PIODevicePlugNotification).
	D2PIO_PTR pContextInfo				//Context passed into D2PIO_RegisterCallbackForDeviceListNotifications().
	);

//D2PIO_RegisterCallbackForDeviceListNotifications() is not currently implemented.
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_RegisterCallbackForDeviceListNotifications(
	D2PIO_LIBRARY_HANDLE hLib,
	gtype_uint32 commTransportId,	//[in] D2PIO_COMM_TRANSPORT_...
	P_D2PIO_DEVICELISTNOTIFICATION_CALLBACK notificationCallbackFunc,
	D2PIO_PTR pContextInfo);

//D2PIO_DeregisterCallbackForDeviceListNotifications() is not currently implemented.
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_DeregisterCallbackForDeviceListNotifications(
	D2PIO_LIBRARY_HANDLE hLib,
	gtype_uint32 commTransportId,	//[in] D2PIO_COMM_TRANSPORT_...
	P_D2PIO_DEVICELISTNOTIFICATION_CALLBACK notificationCallbackFunc);

/***************************************************************************************************************************
Function Name: D2PIO_EnableAutoDeviceDiscoveryAcrossCommTransport()

Purpose:	Calling this function tells the library to automatically search for new devices attached to the
			system via the specified comm transport. Generally the D2PIO library flushes the internal list of discovered
			devices when this function is called and then proceeds to add new devices to the list as they are discovered.

			Currently D2PIO_EnableAutoDeviceDiscoveryAcrossCommTransport() only supports D2PIO_COMM_TRANSPORT_BLUETOOTH_LE.
			Use D2PIO_SearchForDevices() to update the list for D2PIO_COMM_TRANSPORT_USB.

			Call D2PIO_OpenDeviceListSnapshot() to access the list of discovered devices.

Return:		0 iff successful, else < 0. Possible errors include D2PIO_ERROR_BLE_RADIO_NOT_AVAILABLE.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_EnableAutoDeviceDiscoveryAcrossCommTransport(
	D2PIO_LIBRARY_HANDLE hLib,	//[in] handle returned by D2PIO_Init()
	gtype_uint32 deviceType,		//[in] D2PIO_DEVTYPE_...
	gtype_uint32 commTransportId);	//[in] D2PIO_COMM_TRANSPORT_...

/***************************************************************************************************************************
Function Name: D2PIO_DisableAutoDeviceDiscoveryAcrossCommTransport()

Purpose:	Calling this function tells the library to stop searching for new devices attached to
			system via the specified comm transport.

Return:		0 iff successful, else < 0. Possible errors include D2PIO_ERROR_BLE_RADIO_NOT_AVAILABLE.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_DisableAutoDeviceDiscoveryAcrossCommTransport(
	D2PIO_LIBRARY_HANDLE hLib,		//[in] handle returned by D2PIO_Init()
	gtype_uint32 deviceType,		//[in] D2PIO_DEVTYPE_...
	gtype_uint32 commTransportId);	//[in] D2PIO_COMM_TRANSPORT_...

/***************************************************************************************************************************
Function Name: D2PIO_IsAutoDeviceDiscoveryEnabledAcrossCommTransport()

Purpose:	Indicate if the library is looking for devices of a specified type on the specified comm transport.

Return:		1 if device discovery is enabled, else 0

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_bool D2PIO_IsAutoDeviceDiscoveryEnabledAcrossCommTransport(
	D2PIO_LIBRARY_HANDLE hLib,		//[in] handle returned by D2PIO_Init()
	gtype_uint32 deviceType,		//[in] D2PIO_DEVTYPE_...
	gtype_uint32 commTransportId);	//[in] D2PIO_COMM_TRANSPORT_...

/***************************************************************************************************************************
Function Name: D2PIO_OpenDeviceListSnapshot()

Purpose:	The D2PIO library maintains a separate list of available devices for each supported device type.
			Use D2PIO_SearchForDevices() to update the list for D2PIO_COMM_TRANSPORT_USB.
			Use D2PIO_EnableAutoDeviceDiscoveryAcrossCommTransport() to update the list for D2PIO_COMM_TRANSPORT_BLUETOOTH_LE.

			D2PIO_OpenDeviceListSnapshot() creates a snapshot copy of the device list for the specified device type.
			Use the handle returned by D2PIO_OpenDeviceListSnapshot() to walk the device list.

			A device is placed in the list snapshot even if it has already been opened.

Return:		handle to device list snapshot iff successful, else NULL.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL D2PIO_DEVICE_LIST_HANDLE D2PIO_OpenDeviceListSnapshot(
	D2PIO_LIBRARY_HANDLE hLib,			//[in] handle returned by D2PIO_Init()
	gtype_uint32 deviceType,			//[in] D2PIO_DEVTYPE_...
	gtype_uint32 commTransportId,		//[in] D2PIO_COMM_TRANSPORT_...
	gtype_uint32 *pNumDevices,			//[out] ptr to storage loc for the number of devices in the list
	gtype_uint32 *pDeviceListSignature);//[out] ptr to device list signature. Signatures are not fully implemented yet.

/***************************************************************************************************************************
Function Name: D2PIO_CloseDeviceListSnapshot()

Purpose:	Close the list snapshot created by D2PIO_OpenDeviceListSnapshot().

Return:		0 iff successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_CloseDeviceListSnapshot(D2PIO_DEVICE_LIST_HANDLE hDeviceList);//[i] handle returned by D2PIO_OpenDeviceListSnapshot().

/***************************************************************************************************************************
Function Name: D2PIO_DeviceListSnapshot_GetNthEntry()

Purpose:	Return the Nth entry in the list created by D2PIO_OpenDeviceListSnapshot().
			A device is placed in the list snapshot even if it has already been opened.

			Pass the device name string placed in *pDevnameBuf to D2PIO_Device_Open() to open the device. Each
			device name string uniquely identifies the device, so you can determine if a device is already open
			by comparing *pDevnameBuf with the string returned by D2PIO_Device_GetOpenDeviceName() for each open
			device handle.

Return:		0 iff successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_DeviceListSnapshot_GetNthEntry(
	D2PIO_DEVICE_LIST_HANDLE hDeviceList,	//[in] handle returned by D2PIO_OpenDeviceListSnapshot()
	gtype_uint32 N,		//[in] index into list of known devices, 0 => first device in list.
	char *pDevnameBuf,	//[out] ptr to buffer to store NULL terminated UTF-8 string that uniquely identifies the device. Pass this to D2PIO_Device_Open().
	gtype_uint32 DevnameBufsize,//[in] number of bytes in buffer pointed to by pDevnameBuf. strlen(pDevnameBuf) < bufSize, because the string is NULL terminated.
						//strlen(pDevnameBuf)is guaranteed to be < D2PIO_MAX_SIZE_DEVICE_NAME.
	char *pDevFriendlyNameBuf,	//[out] ptr to buffer to store NULL terminated UTF-8 string that may be displayed for the user to identify the device.
						//Note that pDevFriendlyNameBuf is not very friendly for D2PIO_COMM_TRANSPORT_USB.
						//However, the friendly name can be obtained via D2PIO_Device_GetOpenDeviceName() after the device is open.
	gtype_uint32 DevFriendlyNameBufsize,//[in] number of UTF-8 chars in buffer pointed to by pDevFriendlyNameBuf. strlen(pDevFriendlyNameBuf) < bufSize, 
						//because the string is NULL terminated.
						//strlen(pDevFriendlyNameBuf)is guaranteed to be < D2PIO_MAX_SIZE_DEVICE_NAME.
	gtype_int32 *pRSSI_level);	//[out] Valid RSSI levels are < 0. 0 is an invalid RSSI level. This output is only set for D2PIO_COMM_TRANSPORT_BLUETOOTH_LE.


/**********************************************************************************************************************************************************************/

/***************************************************************************************************************************
Function Name: D2PIO_Device_Open()

Purpose:	Open a device with the name returned by D2PIO_DeviceListSnapshot_GetNthEntry.
	
			D2PIO_Device_Open() returns immediately with a non NULL handle if successful.
			Call D2PIO_Device_GetOpenStatus() or wait for D2PIODeviceNotificationOpenStatus to be reported via the supplied
			callback function until device_open_status is not D2PIO_DEVICE_OPEN_STATUS_IN_PROGRESS.

			In theory, once the device_open_status is D2PIO_DEVICE_OPEN_STATUS_SUCCESS, the library knows basically
			everything that it needs to know about the open device. Query functions such as
			D2PIO_GetMeasurementChannelAvailabilityMask() and D2PIO_Device_GetMeasurementChannelInfo() allow client
			applications to discover device details. These query functions generally run almost instantaneously
			because the info that they are supplying to the caller was cached in local memory by D2PIO_Device_Open().

			After calling D2PIO_Device_Open() and waiting for D2PIO_DEVICE_OPEN_STATUS_SUCCESS, an application needs to perform the 
			following operations to take measurements:

			-	Call D2PIO_Device_SetMeasurementPeriod() to set the sampling period used for periodic measurements.
			-	Send a D2PIO_CMD_ID_START_MEASUREMENTS command to start taking measurements.
			-	Call D2PIO_Device_ReadMeasurements(channel) to retrieve measurements.
			-	Send a D2PIO_CMD_ID_STOP_MEASUREMENTS command to stop taking measurements. Once D2PIO_CMD_ID_STOP_MEASUREMENTS
				has been sent, the app can take its time about calling D2PIO_Device_ReadMeasurements() to pull measurements
				out of the measurement buffers. However, the app must empty the measurement buffers before sending
				D2PIO_CMD_ID_START_MEASUREMENTS again.

			An more detailed overview of this process is demonstrated in the D2PIO_DeviceCheck sample application.

			D2PIO_Device_Open() primarily opens devices of the type D2PIO_DEVTYPE_GENERIC, which corresponds to any of the
			standard Go Direct devices from Vernier.
			If the D2PIO library was built with the macro SUPPORT_GO_WIRELESS defined, then D2PIO_Device_Open() can also open
			devices of the type D2PIO_DEVTYPE_GO_WIRELESS_TEMPERATURE, D2PIO_DEVTYPE_GO_WIRELESS_PH, D2PIO_DEVTYPE_GO_WIRELESS_EA,
			and D2PIO_DEVTYPE_POLAR_HEART_RATE.

			The Go Direct devices support the command protocol documented in D2PIOSourceCmds.h .
			The Go Wireless devices support a small subset of the protocol. Basically just enough to take measurements.
			The command protocol is implemented using D2PIO_Device_SendCmdAndGetResponse().

			At any given time, a device is 'owned' by only one thread. The thread that calls D2PIO_Device_Open() is the
			initial owner of the device. If a D2PIO() call is made from a thread that does not own the device object
			that is passed in, then the call will generally fail. To allow another thread to access a device,
			the owning thread must call D2PIO_Device_Unlock(), and then the new thread should call D2PIO_Device_Lock().

			Note that the supplied function notificationCallbackFunc runs on a thread internal to D2PIO library.
			No assumptions should be made about this execution context. In particular, the code supplied in
			notificationCallbackFunc should be very 'light weight' and it should never block. This function should always
			return in less than 10 milliseconds. Failure to meet this constraint can break D2PIO lib.
			Basically this routine should post a message and return immediately. You should not call D2PIO functions in this
			callback, and if you do so, they will generally fail.

Return:		handle to open device if successful, else NULL.

****************************************************************************************************************************/

typedef gtype_int32 (*P_D2PIO_DEVICENOTIFICATION_CALLBACK)
(
	D2PIO_PTR pNotification,			//Eg ptr to D2PIODeviceNotificationOpenStatus.
	gtype_uint32 numBytes,				//Eg sizeof(D2PIODeviceNotificationOpenStatus).
	D2PIO_DEVICE_HANDLE hDevice,		//Uniquely identifies a device until D2PIO_Uninit() is called. This is the same handle returned by D2PIO_Device_Open();
	D2PIO_PTR pContextInfo				//Context passed into D2PIO_Device_Open().
);

D2PIO_LIB_INTERFACE_DECL D2PIO_DEVICE_HANDLE D2PIO_Device_Open(
	D2PIO_LIBRARY_HANDLE hLib,
	const char *pDeviceName,	//[in] NULL terminated UTF-8 string that uniquely identifies the device. See D2PIO_DeviceListSnapshot_GetNthEntry().
	D2PIO_PTR parmsPtr,			//[in] ptr to open configuration parms
	gtype_uint32 parmsLen,		//[in] length of open configuration parms. (parmsLen == sizeof(gtype_bool)) => parmsPtr points to (gtype_bool ReadNVMemoryFlag).
	gtype_uint32 timeoutMs,		//[in] D2PIO_Device_Open() returns immediately. Call D2PIO_Device_GetOpenStatus() to see how Open is progressing.
	P_D2PIO_DEVICENOTIFICATION_CALLBACK notificationCallbackFunc, //ptr to notification callback, used for various device notifications including open status changes. May be NULL.
	D2PIO_PTR pContextInfo);										//ptr passed back via notificationCallbackFunc

#define D2PIO_DEVICE_OPEN_STATUS_ERROR 1
#define D2PIO_DEVICE_OPEN_STATUS_IN_PROGRESS 2
#define D2PIO_DEVICE_OPEN_STATUS_TIMEOUT 3
#define D2PIO_DEVICE_OPEN_STATUS_SUCCESS 4
#define D2PIO_DEVICE_OPEN_STATUS_CLOSED 5
#define D2PIO_DEVICE_OPEN_STATUS_ERROR_BLUETOOTH_NOT_PAIRED 6

/*********************************************************************************************************************************************************************/
//Device notification messages reported via notificationCallbackFunc passed into D2PIO_Device_Open().
#if defined(TARGET_OS_WIN) || defined(TARGET_OS_D2P)
#pragma pack(push)
#pragma pack(1)
#endif

#ifdef TARGET_OS_MAC
#pragma pack(1)
#endif

#define D2PIO_NOTIFY_TYPE_DEVICE_OPEN_STATUS 20
#define D2PIO_NOTIFY_TYPE_DEVICE_CMD_RESPONSE_RECEIVED 25
#define D2PIO_NOTIFY_TYPE_DEVICE_DROPPED_MEASUREMENT 26

struct tagD2PIODeviceNotificationGeneric
{
	unsigned char notificationType;	//D2PIO_NOTIFY_TYPE_DEVICE_...
	unsigned char payload[1];		//Different payloads are defined for different notificationType's.
};
typedef struct tagD2PIODeviceNotificationGeneric D2PIODeviceNotificationGeneric;//Used with D2PIO_NOTIFY_TYPE_DEVICE_CMD_RESPONSE_RECEIVED(empty payload).

struct tagD2PIODeviceNotificationOpenStatus
{
	unsigned char notificationType;	//D2PIO_NOTIFY_TYPE_DEVICE_OPEN_STATUS
	unsigned char device_open_status;// D2PIO_DEVICE_OPEN_STATUS_...
};
typedef struct tagD2PIODeviceNotificationOpenStatus D2PIODeviceNotificationOpenStatus;//Used with D2PIO_NOTIFY_TYPE_DEVICE_OPEN_STATUS.

struct tagD2PIODeviceNotificationDroppedMeasurement
{
	unsigned char notificationType;	//D2PIO_NOTIFY_TYPE_DEVICE_DROPPED_MEASUREMENT
	D2PIO_LittleEndian_uint64 timestamp;  //Microseconds since Jan 1 1970 Coordinated Universal Time(aka UTC aka Greenwich Mean Time).
};
typedef struct tagD2PIODeviceNotificationDroppedMeasurement D2PIODeviceNotificationDroppedMeasurement;//Used with D2PIO_NOTIFY_TYPE_DEVICE_DROPPED_MEASUREMENT.

#if defined(TARGET_OS_WIN) || defined(TARGET_OS_D2P)
#pragma pack(pop)
#endif

#ifdef TARGET_OS_MAC
#pragma pack()
#endif

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetOpenStatus()

Purpose:	Determine that status of the most recent D2PIO_Device_Open() call.

			D2PIO_Device_Open() returns immediately with a non NULL handle if successful.
			Call D2PIO_Device_GetOpenStatus() or wait for D2PIODeviceNotificationOpenStatus to be reported via the supplied
			callback function until device_open_status is not D2PIO_DEVICE_OPEN_STATUS_IN_PROGRESS.

Return:		0 iff successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetOpenStatus(
	D2PIO_DEVICE_HANDLE hDevice,//[in] handle to open device.
	gtype_int32 *pDeviceOpenStatus);//[out] D2PIO_DEVICE_OPEN_STATUS_...

/***************************************************************************************************************************
Function Name: D2PIO_Device_Close()

Purpose:	Close a specified device. After this routine runs, the hDevice handle is no longer valid.
			If D2PIO_Device_Open() returns a non NULL handle, then you should always call D2PIO_Device_Close(handle) when
			you are done with the device, even if an error occurred.

			D2PIO_Device_Close() must be called from the thread that currently owns the device.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_Close(
	D2PIO_DEVICE_HANDLE hDevice);//[in] handle to open device.

/***************************************************************************************************************************
Function Name: D2PIO_GetMeasurementChannelAvailabilityMask()

Purpose:	Retrieve a bit mask that identifies which channels are available for taking measurements.

			Bit 0 (the least significant bit) is used to identify channel 0.
			Bit 1 is used to identify channel 1, and so on.
			The D2PIO_SingleChannelMask(channel) macro converts a channel number to a channel bitmask.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
#define D2PIO_IsChannelActive(channel, mask)  (((1 << ((unsigned int) channel)) & mask) != 0)
#define D2PIO_SingleChannelMask(channel) (1 << ((unsigned int) channel))

D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_GetMeasurementChannelAvailabilityMask(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	gtype_uint32 *pChannelAvailabilityMask);//[out] ptr to loc to store mask identifying which channels are supported by the device

/***************************************************************************************************************************
Function Name: D2PIO_GetMeasurementChannelDefaultActivationMask()

Purpose:	Retrieve a bit mask that identifies which channels are recommended for taking measurements in most situations.
			This mask is a subset of the mask returned by D2PIO_GetMeasurementChannelAvailabilityMask().

			Bit 0 (the least significant bit) is used to identify channel 0.
			Bit 1 is used to identify channel 1, and so on.
			The D2PIO_SingleChannelMask(channel) macro converts a channel number to a channel bitmask.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_GetMeasurementChannelDefaultActivationMask(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	gtype_uint32 *pChannelDefaultActivationMask);//[out] ptr to loc to store mask identifying which channels are recommended for taking measurements by default.

/***************************************************************************************************************************
Function Name: D2PIO_GetMeasurementChannelDefaultActivationMask()

Purpose:	Store permanently on the sensor a bit mask that identifies which channels are recommended for taking measurements 
			in most situations. This mask is a subset of the mask returned by D2PIO_GetMeasurementChannelAvailabilityMask(),
			so it is effectively AND'ed with the mask returned from D2PIO_GetMeasurementChannelAvailabilityMask().

			Bit 0 (the least significant bit) is used to identify channel 0.
			Bit 1 is used to identify channel 1, and so on.
			The D2PIO_SingleChannelMask(channel) macro converts a channel number to a channel bitmask.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_SetMeasurementChannelDefaultActivationMask(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	gtype_uint32 channelDefaultActivationMask,//[in] mask identifying which channels are recommended for taking measurements by default.
	gtype_uint32 timeoutMs);		//[in] # of milliseconds to wait for a reply before giving up. D2PIO_TIMEOUT_MS_DEFAULT is recommended.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetOpenDeviceName()

Purpose:	Get the unique device name string for a specified open device. This is the string returned by
			D2PIO_DeviceListSnapshot_GetNthEntry() and passed in to D2PIO_Device_Open().

			This routine also returns the 'friendly' name for the device which may be displayed for the user to identify the device. 

Return:		0 iff successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetOpenDeviceName(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	char *pDeviceName,	//[out] ptr to buffer to store NULL terminated UTF-8 string that uniquely identifies the device. This was passed to D2PIO_Device_Open(). May be NULL.
	gtype_uint32 DevnameBufsize,//[in] number of bytes in buffer pointed to by pDeviceName. strlen(pDeviceName) < bufSize, because the string is NULL terminated.
	char *pDevFriendlyNameBuf,	//[out] ptr to buffer to store NULL terminated UTF-8 string that may be displayed for the user to identify the device. May be NULL.
	gtype_uint32 DevFriendlyNameBufsize);//[in] number of UTF-8 chars in buffer pointed to by pDevFriendlyNameBuf. strlen(pDevFriendlyNameBuf) < bufSize, because the string is NULL terminated.
										//strlen(pDevFriendlyNameBuf)is guaranteed to be < D2PIO_MAX_SIZE_DEVICE_NAME.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetDeviceDescription()

Purpose:	Get the device description string for the open device.

Return:		0 iff successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetDeviceDescription(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	char *pDeviceDescription,		//[out] ptr to buffer to store NULL UTF-8 terminated string that describes the device.
	gtype_uint32 DescriptionBufsize);//[in] number of bytes in buffer pointed to by pDeviceDescription. strlen(pDeviceDescription) < bufSize, because the string is NULL terminated.
										 //strlen(pDeviceDescription)is guaranteed to be < D2PIO_MAX_SIZE_DEVICE_NAME.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetOpenDeviceType()

Purpose:	Query open device for device type.

Return:		D2PIO_DEVTYPE_... if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetOpenDeviceType( //Returns D2PIO_DEVTYPE_... if successful, else -1
	D2PIO_DEVICE_HANDLE hDevice);	//[in] handle to open device.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetOrderCode()

Purpose:	Get the OrderCode string for the open device.

Return:		0 iff successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetOrderCode(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	char *pOrderCode,				//[out] ptr to buffer to store NULL terminated UTF-8 string for the OrderCode.
	gtype_uint32 Bufsize);			//[in] number of UTF-8 chars in buffer pointed to by pOrderCode. strlen(pOrderCode) < bufSize, because the string is NULL terminated.
									//strlen(pOrderCode) is guaranteed to be < D2PIO_MAX_ORDERCODE_LENGTH.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetSerialNumber()

Purpose:	Get the SerialNumber string for the open device.

Return:		0 iff successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetSerialNumber(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	char *pSerialNumber,			//[out] ptr to buffer to store NULL terminated UTF-8 string for the SerialNumber.
	gtype_uint32 Bufsize);			//[in] number of UTF-8 chars in buffer pointed to by pSerialNumber. strlen(pSerialNumber) < bufSize, because the string is NULL terminated.
									//strlen(pSerialNumber) is guaranteed to be < D2PIO_MAX_SERIALNUMBER_LENGTH.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetManufacturingInfo()

Purpose:	Get the manufacturing info for the open device.

Return:		0 iff successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetManufacturingInfo(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	gtype_uint16 *pManufacturerId,	//[out]
	gtype_uint16 *pManufacturedYear,//[out]
	unsigned char *pManufacturedMonth,//[out]
	unsigned char *pManufacturedDay);//[out] day of month

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetFirmwareVersionInfo()

Purpose:	Get the firmware version info for the open device.

Return:		0 iff successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetFirmwareVersionInfo(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	unsigned char *pMajorVersionMasterCPU, //[out] Binary coded decimal
	unsigned char *pMinorVersionMasterCPU, //[out] Binary coded decimal
	gtype_uint16 *pBuildNumMasterCPU, //[out]
	unsigned char *pMajorVersionSlaveCPU, //[out] Binary coded decimal
	unsigned char *pMinorVersionSlaveCPU, //[out] Binary coded decimal
	gtype_uint16 *pBuildNumSlaveCPU); //[out]

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetBLE_Address()

Purpose:	Get the Bluetooth Low Energy address for the open device.

Return:		0 iff successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetBLE_Address(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	unsigned char *pBLE_Address,	//[out] ptr to buffer to store array of raw binary bytes for BLE address. 
									//This is not NULL terminated! Displays like this --> [5]:[4]:[3]:[2]:[1]:[0] .
	gtype_uint32 Bufsize);			//[in] size of buffer pointed at by pBLE_Address.  Bufsize should be >= D2PIO_BLE_ADDRESS_LENGTH .

/***************************************************************************************************************************
Function Name: D2PIO_Device_Lock()

Purpose:	Lock a specified device so that no other thread can access it. This routine will fail if the device
			is currently locked by another thread.

			D2PIO_Device_Lock() increments the lock count for the specified device by 1. In order for a second
			thread to acquire access to the device, the first thread must call D2PIO_Device_Unlock() until the lock
			count reaches 0. Note that the lock count is set to 1 by D2PIO_Device_Open().

			It is ok for the lock count to be greater than 1 when D2PIO_Device_Close() is called.

			This is an advanced routine that most users should not use. It is only necessary if an application
			needs to access a single device from multiple threads.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_Lock(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	gtype_uint32 timeoutMs);	//[in] wait this long to acquire the lock before failing.

/***************************************************************************************************************************
Function Name: D2PIO_Device_Unlock()

Purpose:	Decrement the lock count for the specified device by 1. This routine will fail if the device
			is not currently locked by the calling thread.

			In order for a second thread to acquire access to the device, the first thread must call
			D2PIO_Device_Unlock() until the lock count reaches 0. Note that the lock count is set to 1 by
			D2PIO_Device_Open().

			If the lock count reaches zero, it is important to make a call to D2PIO_Device_Lock() to lock the
			device. This will prevent multiple threads from simultaneously accessing the device, which can cause
			unpredictable results.

			It is ok for the lock count to be greater than 1 when D2PIO_Device_Close() is called.

			This is an advanced routine that most users should not use. It is only necessary if an application
			needs to access a single device from multiple threads.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_Unlock(
	D2PIO_DEVICE_HANDLE hDevice);//[in] handle to open device.

/***************************************************************************************************************************
Function Name: D2PIO_Device_ClearIO()

Purpose:	Clear the input/output communication buffers for a specified device.
			This also expties the D2PIO Measurement Buffer associated with the specified channel, which holds measurements
			that have been retrieved from the device.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_ClearIO(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel);		//[in] -1 => all channels.

/***************************************************************************************************************************
Function Name: D2PIO_Device_SendCmdAndGetResponse()

Purpose:	Send a command to the specified device hardware and wait for a response.

			Each device type has a command protocol that is unique to that device type. 
			The Go Direct device (D2PIO_DEVTYPE_GENERIC) protocol is documented in D2PIOSourceCmds.h .

			Note that if you send D2PIO_CMD_ID_START_MEASUREMENTS while D2PIO_Device_GetNumMeasurementsAvailable()
			says measurements are available, then the internal measurement buffers managed by the D2PIO library will
			be flushed so that any measurements retrieved sebsequently will have come in after D2PIO_CMD_ID_START_MEASUREMENTS
			was sent.

			Every command supported by D2PIO_Device_SendCmdAndGetResponse() has an associated response. If no response
			specific to a command is defined, then the format of the response is D2PIODefaultCmdResponse. Some commands
			have associated parameter blocks defined for them. If the caller is not concerned about the contents
			of the response for a command, he may pass in NULL for pRespBuf. This is reasonable because the return
			code for D2PIO_Device_SendCmdAndGetResponse() will indicate success or failure. Even if NULL is
			passed in for pRespBuf, D2PIO_Device_SendCmdAndGetResponse() always waits for a response to come back
			from the device. If no response is received after timeoutMs milliseconds, then D2PIO_Device_SendCmdAndGetResponse()
			will return with and error code.

			If D2PIO_Device_SendCmdAndGetResponse() returns -1 and (1 == *pnRespBytes), then *pRespBuf contains
			D2PIODefaultCmdResponse, even if a different response structure is defined for the command.
			The D2PIODefaultCmdResponse structure contains only a single status byte field, which contains
			a D2PIO_STATUS_ERROR_... value. Additional information about a D2PIO_Device_SendCmdAndGetResponse() error may be obtained 
			by calling D2PIO_Device_GetLastCmdResponseStatus().

			Errors coming back from D2PIO_Device_SendCmdAndGetResponse() may or may not be fatal. 
			By definition errors are fatal if they cause cause D2PIO_Device_GetOpenStatus() to return 
			D2PIO_DEVICE_OPEN_STATUS_ERROR or D2PIO_DEVICE_OPEN_STATUS_TIMEOUT.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_SendCmdAndGetResponse(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	unsigned char cmd,			//[in] command code.
	D2PIO_PTR pParams,			//[in] ptr to cmd specific parameter block, may be NULL. See GSkipCommExt.h.
	gtype_uint32 nParamBytes,	//[in] # of bytes in (*pParams).
	D2PIO_PTR pRespBuf,			//[out] ptr to destination buffer, may be NULL.
	gtype_uint32 *pnRespBytes,	//[in, out] ptr to size of of pRespBuf buffer on input, size of response on output, should be NULL if pRespBuf is NULL.
	gtype_uint32 timeoutMs);	//[in] # of milliseconds to wait for a reply before giving up. Most devices should reply to almost all the 
							//currently defined commands within D2PIO_TIMEOUT_MS_DEFAULT milliseconds. In fact, typical response
							//times are less than 50 milliseconds. See D2PIO_TIMEOUT_MS_* definitions.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetLastCmdResponseStatus()

Purpose:	Get error information for the device.

			In principle, any command sent to the device can result in an error.
			If D2PIO_Device_SendCmdAndGetResponse() ever fails and returns a non-zero return code,
			you can gather additional info about what went wrong by calling D2PIO_Device_GetLastCmdResponseStatus().

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetLastCmdResponseStatus(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	unsigned char *pLastCmd,	//[out] last cmd sent to the device
	unsigned char *pLastCmdStatus,//[out] status of last command sent to the device.
								  //If command ran successfully and the device reported good status, then this will be be D2PIO_STATUS_SUCCESS(aka 0).
								  //If no response has been reported back from the device, then this will be D2PIO_STATUS_ERROR_COMMUNICATION.
								  //If the device reported a failure, then this will be a cmd specific error, eg D2PIO_STATUS_ERROR...
	unsigned char *pLastCmdWithErrorRespSentOvertheWire, //[out] last cmd sent that caused the device to report back an error.
	unsigned char *pLastErrorSentOvertheWire);//[out] last error that came back from the device 'over the wire'.

/***************************************************************************************************************************
Function Name: D2PIO_Device_SendCmd()

Purpose:	D2PIO_Device_SendCmd() is an advanced function. You should usually use
			D2PIO_Device_SendCmdAndGetResponse() instead. After calling D2PIO_Device_SendCmd(), you must call
			D2PIO_Device_GetNextResponse() before sending any more commands to the device.

			The D2PIO library will invoke the notificationCallbackFunc passed in to D2PIO_Device_Open() when a
			command response is received. The pNotification argument to notificationCallbackFunc will contain
			D2PIODeviceNotificationGeneric.notificationType = D2PIO_NOTIFY_TYPE_DEVICE_CMD_RESPONSE_RECEIVED.
			This tells the calling app that a response is available so the app should call D2PIO_Device_GetNextResponse().

			Note that calling app should NOT call D2PIO_Device_GetNextResponse() inside notificationCallbackFunc().
			See D2PIO_Device_Open() notes.

			If a response did not come back from the device within the timeoutMs passed in to last D2PIO_Device_SendCmd(), then
			the library will report D2PIO_NOTIFY_TYPE_DEVICE_CMD_RESPONSE_RECEIVED via notificationCallbackFunc and then
			D2PIO_Device_GetNextResponse() will report D2PIODefaultCmdResponse.status=D2PIO_STATUS_ERROR_TIMEOUT in *pRespBuf.

Return:		0 if successful, else < 0. 

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_SendCmd(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	unsigned char cmd,	//[in] command code
	D2PIO_PTR pParams,	//[in] ptr to cmd specific parameter block, may be NULL.
	gtype_uint32 nParamBytes,//[in] # of bytes in (*pParams).
	gtype_uint32 *pSignature,//[out] if not NULL, then put command signature emebedded in cmd blob here.
	gtype_uint32 timeoutMs);	//Num of milliseconds to await response before timing out.

/***************************************************************************************************************************
	Function Name: D2PIO_Device_GetNextResponse()
	
	Purpose:	D2PIO_Device_GetNextResponse() is an advanced function. You should usually use 
				D2PIO_Device_SendCmdAndGetResponse() instead. After calling D2PIO_Device_SendCmd(), you must call
				D2PIO_Device_GetNextResponse() before sending any more commands to the device.

				If D2PIO_Device_GetNextResponse() returns 0 and (1 = *pErrRespFlag) and (1 == *pnRespBytes), 
				then *pRespBuf contains D2PIODefaultCmdResponse, even if a different response structure is defined for the 
				command.

	Return:		0 if successful, else < 0. Fails with return value D2PIO_OPERATION_IN_PROGRESS(-200) if response is not immediately available.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetNextResponse(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	D2PIO_PTR pRespBuf,			//[out] ptr to destination buffer, may be NULL.
	gtype_uint32 *pnRespBytes,  //[in, out] size of of dest buffer on input, size of response on output, should be NULL if pRespBuf is NULL.
	unsigned char *pCmd,		//[out] identifies which command this response is for. Ptr must NOT be NULL!
	gtype_bool *pErrRespFlag,	//[out] flag indicating that the response contains error info. Ptr must NOT be NULL!
	gtype_uint32 *pSignature);	//[in] if not NULL, then compare command signature emebedded in response blob with *pSignature.

/***************************************************************************************************************************
Function Name: D2PIO_Device_WaitForNextResponse()

Purpose:	D2PIO_Device_WaitForNextResponse() is an advanced function. You should usually use
			D2PIO_Device_SendCmdAndGetResponse() instead. 

			D2PIO_Device_WaitForNextResponse() blocks and waits for the next command response to come back from the device.
			If a response does not come back from the device within the timeoutMs passed in to last D2PIO_Device_SendCmd(), then
			D2PIO_Device_WaitForNextResponse() will return immediately after the timeout is detected.

			After calling D2PIO_Device_WaitForNextResponse(), applications should call D2PIO_Device_GetNextResponse().

			If a response did not come back from the device within the timeoutMs passed in to last D2PIO_Device_SendCmd(), then
			D2PIO_Device_GetNextResponse() will report D2PIODefaultCmdResponse.status=D2PIO_STATUS_ERROR_TIMEOUT in *pRespBuf.

Return:		0 if successful, else < 0.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_WaitForNextResponse(
	D2PIO_DEVICE_HANDLE hDevice);	//[in] handle to open device.

/***************************************************************************************************************************
	Function Name: D2PIO_Device_GetMeasurementTick()
	
	Purpose:	The measurement period for devices is specified in discrete 'ticks', so the actual time between 
				measurements is an integer multiple of the tick time.

				Measurement timestamps are also reported in 'ticks'.
	
	Return:		If hDevice is not valid, then this routine returns -1.0, else the return value = the length of time
				in seconds between ticks.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_real64 D2PIO_Device_GetMeasurementTick(
	D2PIO_DEVICE_HANDLE hDevice);	//[in] handle to open device.
	
/***************************************************************************************************************************
Function Name: D2PIO_Device_SetMeasurementPeriod()

Purpose:	This routine sends D2PIO_CMD_ID_SET_MEASUREMENT_PERIOD to the device to set the measurement period for
			the specified channel.

			Note that the measurement period setting only affects channels for which D2PIO_Device_GetMeasurementChannelInfo(channel)
			reports SamplingMode == D2PIO_SAMPLING_MODE_PERIODIC.

			You can find out information about valid measurment periods by calling D2PIO_Device_GetMeasurementChannelPeriodInfo().

			Currently, D2PIO devices only support a single measurement period per device, so the channel argument to
			D2PIO_Device_SetMeasurementPeriod() is ignored. This may change in the future.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_SetMeasurementPeriod(
	D2PIO_DEVICE_HANDLE hDevice,//[in] handle to open device.
	signed char channel,		//[in] -1 => all channels. Currently ignored.
	gtype_real64 desiredPeriod,	//[in] desired measurement period in seconds.
	gtype_uint32 timeoutMs);	//[in] # of milliseconds to wait for a reply before giving up. D2PIO_TIMEOUT_MS_DEFAULT is recommended.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetMeasurementPeriod()

Purpose:	This routine sends D2PIO_CMD_ID_GET_MEASUREMENT_PERIOD to the device to get the measurement period for
			the specified channel.

			Currently, D2PIO devices only support a single measurement period per device, so the channel argument is
			ignored. This may change in the future.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetMeasurementPeriod(
	D2PIO_DEVICE_HANDLE hDevice,//[in] handle to open device.
	signed char channel,		//[in] -1 => return 'base' period, else return period for specified channel. Currently ignored.
	gtype_real64 *pPeriod,		//[out] ptr to loc to store period in seconds.
	gtype_uint32 timeoutMs);	//[in] # of milliseconds to wait for a reply before giving up. D2PIO_TIMEOUT_MS_DEFAULT is recommended.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetNumMeasurementsAvailable()

Purpose:	Report the number of measurements currently stored in the D2PIO Measurement Buffer for the
			specified channel.

			After D2PIO_CMD_ID_START_MEASUREMENTS has been sent to the device, the device starts
			sending measurements to the host computer. These measurements are stored in the
			D2PIO Measurement Buffer. A separate D2PIO Measurement Buffer is maintained for each
			channel of the device.

			Subsequent measurements are sent at the currently configured measurement period interval if the current
			SamplingMode for a channel is D2PIO_SAMPLING_MODE_PERIODIC, not D2PIO_SAMPLING_MODE_APERIODIC.

			If D2PIO_Device_GetMeasurementChannelInfo(channel) reports that the NumericMeasType is D2PIO_NUMERIC_MEAS_TYPE_INT32,
			then call D2PIO_Device_ReadRawMeasurements() to retrieve measurements from the D2PIO Measurement Buffer for the
			specified channel. If NumericMeasType is D2PIO_NUMERIC_MEAS_TYPE_REAL64, then call D2PIO_Device_ReadMeasurements().
			
			The D2PIO Measurement Buffer is guaranteed to hold at least 20000 measurements(per channel).
			The buffer is circular, so if you do not service it often enough, the oldest measurements in the buffer
			are lost. If you wish to capture all the measurements coming from the device, you must call
			D2PIO_Device_ReadRawMeasurements() often enough so that the D2PIO_Device_GetNumMeasurementsAvailable() does not
			reach 20000. On the other hand, we reserve the right to make the Measurement Buffer > 20000 measurements, so
			do not assume that you can empty the buffer simply by reading in 20000 measurements.

			Each of the following actions clears the D2PIO Measurement Buffer for a specified channel:
			1) Call D2PIO_Device_ReadMeasurements() or D2PIO_Device_ReadRawMeasurements() with count set to D2PIO_Device_GetNumMeasurementsAvailable(), 
			or
			2) Call D2PIO_Device_ClearIO().

			The D2PIO Measurement Buffers are generally empty after D2PIO_Device_Open() is called. They do not
			start filling up until D2PIO_CMD_ID_START_MEASUREMENTS is sent to the device. However, if the device was
			previously configured for a 'finite' collection via D2PIO_CMD_ID_SET_FINITE_COLLECTION_PARAMS, then
			D2PIO_Device_Open() will retrieve any measurements stored on the the device and place them in the
			D2PIO Measurement Buffers. 
			
			See discussion of D2PIO_CMD_ID_SET_FINITE_COLLECTION_PARAMS and D2PIO_CMD_ID_GET_FINITE_COLLECTION_STATUS.

			Note that the device stops taking measurements after receiving D2PIO_CMD_ID_STOP_MEASUREMENTS but that 
			does not clear the D2PIO Measurement Buffers.

Return:		number of measurements currently stored in the D2PIO Measurement Buffer for the specified channel.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetNumMeasurementsAvailable(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel);			//[in]

/***************************************************************************************************************************
Function Name: D2PIO_Device_ReadMeasurements()

Purpose:	Retrieve gtype_real64 measurements from the D2PIO Measurement Buffer for a specified channel. The measurements 
			reported by this routine are actually removed from the D2PIO Measurement Buffer.

			After D2PIO_CMD_ID_START_MEASUREMENTS has been sent to the device, the device starts
			sending measurements to the host computer. These measurements are stored in the D2PIO Measurement Buffer. 
			A separate D2PIO Measurement Buffer is maintained for each channel of the device.
			
			If D2PIO_Device_GetMeasurementChannelInfo(channel) reports that the NumericMeasType is D2PIO_NUMERIC_MEAS_TYPE_INT32,
			then call D2PIO_Device_ReadRawMeasurements() to retrieve measurements from the D2PIO Measurement Buffer for the
			specified channel. If NumericMeasType is D2PIO_NUMERIC_MEAS_TYPE_REAL64, then call D2PIO_Device_ReadMeasurements().

			See the description of D2PIO_Device_GetNumMeasurementsAvailable().

Return:		Number of measurements retrieved from the D2PIO Measurement Buffer. This routine
			returns immediately, so the return value may be less than maxCount.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_ReadMeasurements( 
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	gtype_real64 *pMeasurementsBuf,	//[out] ptr to loc to store measurements.
	gtype_int64 *pTimeStamps,		//[out] ptr to loc to store 'tick' resolution timestamps. May be NULL.
	gtype_uint32 maxCount);			//[in] maximum number of measurements to copy to pMeasurementsBuf.

/***************************************************************************************************************************
Function Name: D2PIO_Device_ReadMeasurements()

Purpose:	Retrieve gtype_int32 measurements from the D2PIO Measurement Buffer for a specified channel. The measurements
			reported by this routine are actually removed from the D2PIO Measurement Buffer.

			After D2PIO_CMD_ID_START_MEASUREMENTS has been sent to the device, the device starts
			sending measurements to the host computer. These measurements are stored in the D2PIO Measurement Buffer.
			A separate D2PIO Measurement Buffer is maintained for each channel of the device.

			If D2PIO_Device_GetMeasurementChannelInfo(channel) reports that the NumericMeasType is D2PIO_NUMERIC_MEAS_TYPE_INT32,
			then call D2PIO_Device_ReadRawMeasurements() to retrieve measurements from the D2PIO Measurement Buffer for the
			specified channel. If NumericMeasType is D2PIO_NUMERIC_MEAS_TYPE_REAL64, then call D2PIO_Device_ReadMeasurements().

			See the description of D2PIO_Device_GetNumMeasurementsAvailable().

Return:		Number of measurements retrieved from the D2PIO Measurement Buffer. This routine
			returns immediately, so the return value may be less than maxCount.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_ReadRawMeasurements( //Returns the number of measurements read
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	gtype_int32 *pMeasurementsBuf,	//[out] ptr to loc to store measurements.
	gtype_int64 *pTimeStamps,		//[out] ptr to loc to store 'tick' resolution timestamps. May be NULL.
	gtype_uint32 maxCount);	//[in] maximum number of measurements to copy to pMeasurementsBuf.

D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_CopyLatestMeasurement(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	gtype_real64 *pMeasurement,		//[out] ptr to loc to store measurement.
	gtype_int64 *pTimeStamp);		//[out] ptr to loc to store 'tick' resolution timestamp. May be NULL.

D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_CopyLatestRawMeasurement(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	gtype_int32 *pMeasurement,		//[out] ptr to loc to store measurement.
	gtype_int64 *pTimeStamp);		//[out] ptr to loc to store 'tick' resolution timestamp. May be NULL.

D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetNumMeasurementsAvailableInTimeRange(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	gtype_int64 priorTimestamp,		//[in] Count measurements with timestamps > priorTimestamp
	gtype_int64 finalTimestamp);	//[in] Count measurements with timestamps <= finalTimestamp

D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_CopyMeasurements( //Returns the number of measurements read
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	gtype_int64 priorTimestamp,		//[in] Retrieve measurements with timestamps > priorTimestamp
	gtype_int64 finalTimestamp,		//[in] Retrieve measurements with timestamps <= finalTimestamp
	gtype_real64 *pMeasurementsBuf,	//[out] ptr to loc to store measurements.
	gtype_int64 *pTimeStamps,		//[out] ptr to loc to store 'tick' resolution timestamps. May be NULL.
	gtype_uint32 maxCount);	//[in] maximum number of measurements to copy to pMeasurementsBuf.

D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_CopyRawMeasurements( //Returns the number of measurements read
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	gtype_int64 priorTimestamp,		//[in] Retrieve measurements with timestamps > priorTimestamp
	gtype_int64 finalTimestamp,		//[in] Retrieve measurements with timestamps <= finalTimestamp
	gtype_int32 *pMeasurementsBuf,	//[out] ptr to loc to store measurements.
	gtype_int64 *pTimeStamps,		//[out] ptr to loc to store 'tick' resolution timestamps. May be NULL.
	gtype_uint32 maxCount);	//[in] maximum number of measurements to copy to pMeasurementsBuf.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetMeasurementChannelInfo()

Purpose:	In theory, most of what you need to know about taking measurements from a previously unknown sensor can
			be obtained by calling D2PIO_Device_GetMeasurementChannelInfo(channel) for the channels listed by
			D2PIO_GetMeasurementChannelAvailabilityMask().

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetMeasurementChannelInfo(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	gtype_uint32 *pSensorId,		//[out] sensor id, which is unique record key for the Sensor Map.
	unsigned char *pNumericMeasType, //[out] numeric measurement type, eg D2PIO_NUMERIC_MEAS_TYPE...
	unsigned char *pSamplingMode,    //[out] sampling mode, eg D2PIO_SAMPLING_MODE_...
	char *pSensorDescription,//[out] ptr to loc to store NULL terminated UTF-8 string describing sensor. May be NULL.
	gtype_uint32 sensorDescriptionLen,//[in] number of bytes in buffer pointed to by pSensorDescription. strlen(pSensorDescription) < sensorDescriptionLen, because the string is NULL terminated.
									//strlen(pSensorDescription)is guaranteed to be < D2PIO_MAX_NUM_BYTES_IN_SENSOR_DESCRIPTION.
	char *pSensorUnit,				//[out] ptr to loc to store NULL terminated UTF-8 string describing measurement unit. May be NULL.
	gtype_uint32 sensorUnitLen);	//[in] number of bytes in buffer pointed to by pSensorUnit. strlen(pSensorUnit) < sensorUnitLen, because the string is NULL terminated.
									//strlen(pSensorUnit) is guaranteed to be < D2PIO_MAX_NUM_BYTES_IN_SENSOR_UNIT.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetMeasurementChannelSensorId()

Purpose:	Retrieve the sensor id for the specified channel.
			This information is also available from D2PIO_Device_GetMeasurementChannelInfo().

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetMeasurementChannelSensorId(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	gtype_uint32 *pSensorId);		//[out] sensor id, which is unique record key for the Sensor Map.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetMeasurementChannelNumericType()

Purpose:	Retrieve the numeric measurement type(D2PIO_NUMERIC_MEAS_TYPE_REAL64 or D2PIO_NUMERIC_MEAS_TYPE_INT32) 
			for the specified channel.
			This information is also available from D2PIO_Device_GetMeasurementChannelInfo().

			If the numeric measurement type is D2PIO_NUMERIC_MEAS_TYPE_INT32, then call D2PIO_Device_ReadRawMeasurements() 
			to retrieve measurements from the D2PIO Measurement Buffer for the specified channel. 
			If the numeric measurement type is D2PIO_NUMERIC_MEAS_TYPE_REAL64, then call D2PIO_Device_ReadMeasurements().

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetMeasurementChannelNumericType(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	unsigned char *pNumericMeasType);//[out] numeric measurement type, eg D2PIO_NUMERIC_MEAS_TYPE...

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetMeasurementChannelSamplingMode()

Purpose:	Retrieve the sampling mode(D2PIO_SAMPLING_MODE_PERIODIC or D2PIO_SAMPLING_MODE_APERIODIC)
			for the specified channel.
			This information is also available from D2PIO_Device_GetMeasurementChannelInfo().

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetMeasurementChannelSamplingMode(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	unsigned char *pSamplingMode);	//[out] sampling mode, eg D2PIO_SAMPLING_MODE_...

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetMeasurementChannelSensorDescription()

Purpose:	Retrieve the description string for the sensor in the specified channel.
			This information is also available from D2PIO_Device_GetMeasurementChannelInfo().

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetMeasurementChannelSensorDescription(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	char *pSensorDescription,//[out] ptr to loc to store NULL terminated UTF-8 string describing sensor. 
	gtype_uint32 sensorDescriptionLen);//[in] number of bytes in buffer pointed to by pSensorDescription. strlen(pSensorDescription) < sensorDescriptionLen, because the string is NULL terminated.
									  //strlen(pSensorDescription)is guaranteed to be < D2PIO_MAX_NUM_BYTES_IN_SENSOR_DESCRIPTION.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetMeasurementChannelSensorUnit()

Purpose:	Retrieve the unit string for the sensor in the specified channel.
			This information is also available from D2PIO_Device_GetMeasurementChannelInfo().

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetMeasurementChannelSensorUnit(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	char *pSensorUnit,		//[out] ptr to loc to store NULL terminated UTF-8 string describing measurement unit.
	gtype_uint32 sensorUnitLen);	//[in] number of bytes in buffer pointed to by pSensorUnit. strlen(pSensorUnit) < sensorUnitLen, because the string is NULL terminated.
									//strlen(pSensorUnit) is guaranteed to be < D2PIO_MAX_NUM_BYTES_IN_SENSOR_UNIT.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetMeasurementChannelRangeInfo()

Purpose:	Retrieve the measurement range information for the sensor in the specified channel.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetMeasurementChannelRangeInfo(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	gtype_real64 *pMeasurementUncertainty, //[out] uncertainty of measurement expressed in SensorUnit's.
	gtype_real64 *pMinMeasurement,	//[out] minimum measurement expressed in SensorUnit's.
	gtype_real64 *pMaxMeasurement);	//[out] maximum measurement expressed in SensorUnit's.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetMeasurementChannelPeriodInfo()

Purpose:	Retrieve the measurement period range information for the sensor in the specified channel.
			This information is only valid for sensors whose sampling mode is D2PIO_SAMPLING_MODE_PERIODIC.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetMeasurementChannelPeriodInfo(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	gtype_real64 *pMinMeasurementPeriod,//[out] minimum measurement period in seconds
	gtype_real64 *pMaxMeasurementPeriod,//[out] maximum measurement period in seconds
	gtype_real64 *pTypMeasurementPeriod,//[out] typical measurement period in seconds
	gtype_real64 *pMeasurementPeriodGranularity);//[out] in seconds, supported periods are integer multiples of measurementPeriodGranularity.

/***************************************************************************************************************************
Function Name: D2PIO_Device_GetMeasurementChannelIncompatibilityMask()

Purpose:	Retrieve a mask that identifies which channels cannot collect data concurrently with the channel
			passed in as an argument.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_GetMeasurementChannelIncompatibilityMask(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	signed char channel,			//[in]
	gtype_uint32 *pIncompatibilityMask);//[out] ptr to mask identifying channels incompatible with 'channel' passed in

/***************************************************************************************************************************
Function Name: D2PIO_Device_NVMemBlk_GetAddrRange()

Purpose:	Retrieve the address range in nonvolatile memory on the device that can be read in with D2PIO_Device_NVMemBlk_Read().

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_NVMemBlk_GetAddrRange(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	gtype_uint32 *pMinAddr,		//[out]
	gtype_uint32 *pMaxAddr);	//[out]

/***************************************************************************************************************************
Function Name: D2PIO_Device_NVMemBlk_Read()

Purpose:	Read in the specified range in device nonvolatile memory and place it in the specified destination buffer.

			Note that if the D2PIO_Device_Open() parmsLen argument == sizeof(gtype_bool) and 
			the parmsPtr arg points to (gtype_bool ReadNVMemoryFlag=1), then D2PIO_Device_Open() will read in all the device
			nonvolatile memory and cache it. That allows subsequent calls to D2PIO_Device_NVMemBlk_Read() to read from the
			local cache and return immediately.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_NVMemBlk_Read(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	D2PIO_PTR pDestBuf,
	gtype_uint32 addr,
	gtype_uint32 numBytesToRead,
	gtype_uint32 timeoutMs);		//[in] # of milliseconds to wait for a reply before giving up.

/***************************************************************************************************************************
Function Name: D2PIO_Device_NVMemBlk_Write()

Purpose:	Write out data to the specified range of device nonvolatile memory.
			It is not recommended that third party client applications use this function because it can permanently alter the
			sensor device and cause problems.

Return:		0 if successful, else -1.

****************************************************************************************************************************/
D2PIO_LIB_INTERFACE_DECL gtype_int32 D2PIO_Device_NVMemBlk_Write(
	D2PIO_DEVICE_HANDLE hDevice,	//[in] handle to open device.
	D2PIO_PTR pSrcBuf,
	gtype_uint32 addr,
	gtype_uint32 numBytesToWrite,
	gtype_uint32 timeoutMs);		//[in] # of milliseconds to wait for a reply before giving up.

#endif //_D2PIO_LIB_INTERFACE_H_
